<?php
require_once __DIR__ . '/includes/admin_layout.php';
require_once __DIR__ . '/../includes/settings_store.php';
require_once __DIR__ . '/../includes/toast.php';

$settings = settings_get();
$error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $language = (string)($_POST['language'] ?? 'French');
    $adminEmail = trim((string)($_POST['admin_email'] ?? ''));
    $currentPassword = (string)($_POST['current_password'] ?? '');
    $newPassword = (string)($_POST['new_password'] ?? '');
    $confirmPassword = (string)($_POST['confirm_password'] ?? '');

    $allowedLanguages = ['French', 'English', 'Kinyarwanda'];
    if (!in_array($language, $allowedLanguages, true)) {
        $language = 'French';
    }

    if ($adminEmail === '' || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
        $error = t('settings.error_valid_email');
    } else {
        $existingEmail = (string)($settings['admin_email'] ?? '');
        $existingHash = (string)($settings['admin_password_hash'] ?? '');

        $emailChanged = ($adminEmail !== $existingEmail);
        $passwordChangeRequested = ($newPassword !== '' || $confirmPassword !== '');
        $requiresPassword = ($emailChanged || $passwordChangeRequested);

        if ($requiresPassword) {
            if ($currentPassword === '') {
                $error = t('settings.error_current_required');
            } elseif (!password_verify($currentPassword, $existingHash)) {
                $error = t('settings.error_current_incorrect');
            }
        }

        if ($error === null) {
            $settings['language'] = $language;
            $settings['admin_email'] = $adminEmail;

            if ($passwordChangeRequested) {
                if ($newPassword === '' || $confirmPassword === '') {
                    $error = t('settings.error_fill_both');
                } elseif ($newPassword !== $confirmPassword) {
                    $error = t('settings.error_mismatch');
                } else {
                    $settings['admin_password_hash'] = password_hash($newPassword, PASSWORD_DEFAULT);
                }
            }
        }
    }

    if ($error === null) {
        settings_save($settings);
        toast_set('success', t('settings.updated_success'));
        header('Location: /nyakinama/admin/settings.php');
        exit;
    }
}

admin_header(t('settings.title'));
$year = $_GET['year'] ?? '2024-2025';
?>
<div class="grid grid-cols-1 xl:grid-cols-12 gap-6">
    <section class="xl:col-span-8">
        <div class="rounded-3xl border border-slate-200 bg-white p-6">
            <div class="text-sm font-semibold"><?= htmlspecialchars(t('settings.section_title')) ?></div>
            <div class="mt-1 text-xs text-slate-500"><?= htmlspecialchars(t('settings.subtitle')) ?></div>

            <?php if ($error): ?>
                <div class="mt-5 rounded-2xl border border-red-200 bg-red-50 px-4 py-3 text-sm text-red-900">
                    <div class="font-semibold"><?= htmlspecialchars(t('settings.update_failed')) ?></div>
                    <div class="mt-1 text-sm text-red-800"><?= htmlspecialchars($error) ?></div>
                </div>
            <?php endif; ?>

            <form method="post" class="mt-6 space-y-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-xs font-semibold text-slate-600"><?= htmlspecialchars(t('settings.language')) ?></label>
                        <select class="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-2 text-sm" name="language" required>
                            <?php foreach (['French', 'English', 'Kinyarwanda'] as $lang): ?>
                                <option value="<?= htmlspecialchars($lang) ?>" <?= (($settings['language'] ?? 'French') === $lang) ? 'selected' : '' ?>><?= htmlspecialchars($lang) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div>
                        <label class="block text-xs font-semibold text-slate-600"><?= htmlspecialchars(t('settings.admin_email')) ?></label>
                        <input class="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-2 text-sm" type="email" name="admin_email" value="<?= htmlspecialchars((string)($settings['admin_email'] ?? '')) ?>" required />
                    </div>
                </div>

                <div class="rounded-2xl border border-slate-200 bg-slate-50 p-5">
                    <div class="text-sm font-semibold"><?= htmlspecialchars(t('settings.change_password')) ?></div>
                    <div class="mt-1 text-xs text-slate-600"><?= htmlspecialchars(t('settings.leave_empty')) ?></div>

                    <div class="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-xs font-semibold text-slate-600"><?= htmlspecialchars(t('settings.current_password')) ?></label>
                            <input class="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-2 text-sm" type="password" name="current_password" />
                        </div>
                        <div>
                            <label class="block text-xs font-semibold text-slate-600"><?= htmlspecialchars(t('settings.new_password')) ?></label>
                            <input class="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-2 text-sm" type="password" name="new_password" />
                        </div>
                        <div>
                            <label class="block text-xs font-semibold text-slate-600"><?= htmlspecialchars(t('settings.confirm_password')) ?></label>
                            <input class="mt-1 w-full rounded-xl border border-slate-300 bg-white px-3 py-2 text-sm" type="password" name="confirm_password" />
                        </div>
                    </div>
                </div>

                <div class="flex flex-col sm:flex-row gap-3">
                    <button class="rounded-xl bg-gradient-to-r from-slate-900 to-indigo-900 px-4 py-2 text-sm font-semibold text-white shadow-sm shadow-slate-900/10 hover:from-slate-800 hover:to-indigo-800" type="submit"><?= htmlspecialchars(t('settings.save_changes')) ?></button>
                    <a class="rounded-xl border border-slate-300 bg-white px-4 py-2 text-sm font-semibold hover:bg-slate-50" href="/nyakinama/admin/index.php"><?= htmlspecialchars(t('settings.cancel')) ?></a>
                </div>
            </form>

            <div class="mt-6 text-sm font-semibold"><?= htmlspecialchars(t('settings.categories_management')) ?></div>
            <div class="mt-3 rounded-2xl border border-slate-200 bg-slate-50 p-5 text-sm text-slate-600"><?= htmlspecialchars(t('settings.category_placeholder')) ?></div>

            <div class="mt-6 text-sm font-semibold"><?= htmlspecialchars(t('settings.backup_restore')) ?></div>
            <div class="mt-3 flex flex-col sm:flex-row gap-3">
                <button class="rounded-xl bg-slate-900 px-4 py-2 text-sm font-semibold text-white hover:bg-slate-800" type="button"><?= htmlspecialchars(t('settings.backup')) ?></button>
                <button class="rounded-xl border border-slate-300 bg-white px-4 py-2 text-sm font-semibold hover:bg-slate-50" type="button"><?= htmlspecialchars(t('settings.restore')) ?></button>
            </div>
        </div>
    </section>

    <aside class="xl:col-span-4">
        <div class="rounded-3xl border border-slate-200 bg-white p-6">
            <div class="text-sm font-semibold"><?= htmlspecialchars(t('settings.notes')) ?></div>
            <div class="mt-4 space-y-3 text-sm text-slate-600">
                <div class="rounded-2xl border border-slate-200 bg-slate-50 p-4"><?= htmlspecialchars(t('settings.note_storage')) ?></div>
                <div class="rounded-2xl border border-slate-200 bg-slate-50 p-4"><?= htmlspecialchars(t('settings.note_relogin')) ?></div>
            </div>
        </div>
    </aside>
</div>
<?php
admin_footer();
